'use client';

import React, { useState } from 'react';
import {
    Box,
    Container,
    Typography,
    Grid,
    Card,
    CardContent,
    TextField,
    Button,
    Alert,
} from '@mui/material';
import {
    Email,
    Phone,
    LocationOn,
    Facebook,
    Twitter,
    LinkedIn,
    Instagram,
} from '@mui/icons-material';
import Header from '@/components/Header';
import Footer from '@/components/Footer';

export default function ContactPage() {
    const [formData, setFormData] = useState({
        name: '',
        email: '',
        subject: '',
        message: '',
    });
    const [errors, setErrors] = useState({});
    const [success, setSuccess] = useState(false);

    const handleChange = (e) => {
        const { name, value } = e.target;
        setFormData((prev) => ({ ...prev, [name]: value }));
        if (errors[name]) {
            setErrors((prev) => ({ ...prev, [name]: '' }));
        }
    };

    const validate = () => {
        const newErrors = {};
        if (!formData.name) newErrors.name = 'Name is required';
        if (!formData.email) {
            newErrors.email = 'Email is required';
        } else if (!/\S+@\S+\.\S+/.test(formData.email)) {
            newErrors.email = 'Email is invalid';
        }
        if (!formData.subject) newErrors.subject = 'Subject is required';
        if (!formData.message) newErrors.message = 'Message is required';
        return newErrors;
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        const newErrors = validate();
        if (Object.keys(newErrors).length > 0) {
            setErrors(newErrors);
        } else {
            console.log('Contact form submitted:', formData);
            setSuccess(true);
            setFormData({ name: '', email: '', subject: '', message: '' });
            setTimeout(() => setSuccess(false), 5000);
        }
    };

    const contactInfo = [
        {
            icon: <Email sx={{ fontSize: 40 }} />,
            title: 'Email Us',
            content: 'support@freelancemarketplace.com',
            link: 'mailto:support@freelancemarketplace.com',
        },
        {
            icon: <Phone sx={{ fontSize: 40 }} />,
            title: 'Call Us',
            content: '+1 (555) 123-4567',
            link: 'tel:+15551234567',
        },
        {
            icon: <LocationOn sx={{ fontSize: 40 }} />,
            title: 'Visit Us',
            content: '123 Business St, San Francisco, CA 94107',
            link: '#',
        },
    ];

    const socialLinks = [
        { icon: <Facebook />, name: 'Facebook', link: '#' },
        { icon: <Twitter />, name: 'Twitter', link: '#' },
        { icon: <LinkedIn />, name: 'LinkedIn', link: '#' },
        { icon: <Instagram />, name: 'Instagram', link: '#' },
    ];

    return (
        <>
            <Header />

            {/* Page Header */}
            <Box
                sx={{
                    background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                    color: 'white',
                    py: 8,
                }}
            >
                <Container maxWidth="md" sx={{ textAlign: 'center' }}>
                    <Typography variant="h2" fontWeight={700} gutterBottom>
                        Get In Touch
                    </Typography>
                    <Typography variant="h6" sx={{ opacity: 0.9 }}>
                        We'd love to hear from you. Send us a message and we'll respond as soon as possible.
                    </Typography>
                </Container>
            </Box>

            <Container maxWidth="lg" sx={{ mt: -4, mb: 8 }}>
                {/* Contact Info Cards */}
                <Grid container spacing={3} sx={{ mb: 6 }}>
                    {contactInfo.map((info) => (
                        <Grid item xs={12} md={4} key={info.title}>
                            <Card sx={{ height: '100%', textAlign: 'center', p: 3 }}>
                                <Box
                                    sx={{
                                        width: 80,
                                        height: 80,
                                        borderRadius: '50%',
                                        background: 'linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%)',
                                        display: 'flex',
                                        alignItems: 'center',
                                        justifyContent: 'center',
                                        margin: '0 auto 16px',
                                        color: 'white',
                                    }}
                                >
                                    {info.icon}
                                </Box>
                                <Typography variant="h6" fontWeight={600} gutterBottom>
                                    {info.title}
                                </Typography>
                                <Typography
                                    variant="body1"
                                    color="primary.main"
                                    component="a"
                                    href={info.link}
                                    sx={{ textDecoration: 'none' }}
                                >
                                    {info.content}
                                </Typography>
                            </Card>
                        </Grid>
                    ))}
                </Grid>

                <Grid container spacing={4}>
                    {/* Contact Form */}
                    <Grid item xs={12} md={7}>
                        <Card>
                            <CardContent sx={{ p: 4 }}>
                                <Typography variant="h5" fontWeight={700} gutterBottom>
                                    Send Us a Message
                                </Typography>
                                <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
                                    Fill out the form below and we'll get back to you within 24 hours
                                </Typography>

                                {success && (
                                    <Alert severity="success" sx={{ mb: 3 }}>
                                        Thank you for your message! We'll get back to you soon.
                                    </Alert>
                                )}

                                <form onSubmit={handleSubmit}>
                                    <Grid container spacing={2}>
                                        <Grid item xs={12} sm={6}>
                                            <TextField
                                                fullWidth
                                                label="Your Name"
                                                name="name"
                                                value={formData.name}
                                                onChange={handleChange}
                                                error={!!errors.name}
                                                helperText={errors.name}
                                            />
                                        </Grid>
                                        <Grid item xs={12} sm={6}>
                                            <TextField
                                                fullWidth
                                                label="Your Email"
                                                name="email"
                                                type="email"
                                                value={formData.email}
                                                onChange={handleChange}
                                                error={!!errors.email}
                                                helperText={errors.email}
                                            />
                                        </Grid>
                                        <Grid item xs={12}>
                                            <TextField
                                                fullWidth
                                                label="Subject"
                                                name="subject"
                                                value={formData.subject}
                                                onChange={handleChange}
                                                error={!!errors.subject}
                                                helperText={errors.subject}
                                            />
                                        </Grid>
                                        <Grid item xs={12}>
                                            <TextField
                                                fullWidth
                                                label="Message"
                                                name="message"
                                                multiline
                                                rows={6}
                                                value={formData.message}
                                                onChange={handleChange}
                                                error={!!errors.message}
                                                helperText={errors.message}
                                            />
                                        </Grid>
                                        <Grid item xs={12}>
                                            <Button type="submit" variant="contained" size="large" fullWidth>
                                                Send Message
                                            </Button>
                                        </Grid>
                                    </Grid>
                                </form>
                            </CardContent>
                        </Card>
                    </Grid>

                    {/* Additional Info */}
                    <Grid item xs={12} md={5}>
                        <Card sx={{ mb: 3 }}>
                            <CardContent sx={{ p: 4 }}>
                                <Typography variant="h6" fontWeight={600} gutterBottom>
                                    Office Hours
                                </Typography>
                                <Typography variant="body2" color="text.secondary" paragraph>
                                    Monday - Friday: 9:00 AM - 6:00 PM PST
                                </Typography>
                                <Typography variant="body2" color="text.secondary" paragraph>
                                    Saturday: 10:00 AM - 4:00 PM PST
                                </Typography>
                                <Typography variant="body2" color="text.secondary">
                                    Sunday: Closed
                                </Typography>
                            </CardContent>
                        </Card>

                        <Card sx={{ mb: 3 }}>
                            <CardContent sx={{ p: 4 }}>
                                <Typography variant="h6" fontWeight={600} gutterBottom>
                                    Follow Us
                                </Typography>
                                <Box sx={{ display: 'flex', gap: 2, mt: 2 }}>
                                    {socialLinks.map((social) => (
                                        <Button
                                            key={social.name}
                                            variant="outlined"
                                            sx={{
                                                minWidth: 48,
                                                width: 48,
                                                height: 48,
                                                p: 0,
                                                borderRadius: '50%',
                                            }}
                                            href={social.link}
                                        >
                                            {social.icon}
                                        </Button>
                                    ))}
                                </Box>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardContent sx={{ p: 4 }}>
                                <Typography variant="h6" fontWeight={600} gutterBottom>
                                    Quick Links
                                </Typography>
                                <Box sx={{ display: 'flex', flexDirection: 'column', gap: 1 }}>
                                    <Button variant="text" href="/faq" sx={{ justifyContent: 'flex-start' }}>
                                        FAQ
                                    </Button>
                                    <Button variant="text" href="/how-it-works" sx={{ justifyContent: 'flex-start' }}>
                                        How It Works
                                    </Button>
                                    <Button variant="text" href="/terms" sx={{ justifyContent: 'flex-start' }}>
                                        Terms & Conditions
                                    </Button>
                                    <Button variant="text" href="/privacy" sx={{ justifyContent: 'flex-start' }}>
                                        Privacy Policy
                                    </Button>
                                </Box>
                            </CardContent>
                        </Card>
                    </Grid>
                </Grid>
            </Container>

            <Footer />
        </>
    );
}
