'use client';

import React, { useState } from 'react';
import {
    AppBar,
    Toolbar,
    Box,
    Button,
    IconButton,
    TextField,
    InputAdornment,
    Container,
    useScrollTrigger,
    Drawer,
    List,
    ListItem,
    ListItemButton,
    ListItemText,
} from '@mui/material';
import {
    Search as SearchIcon,
    Menu as MenuIcon,
    Close as CloseIcon,
} from '@mui/icons-material';
import Link from 'next/link';

const navItems = [
    { label: 'Home', href: '/' },
    { label: 'Browse Services', href: '/services' },
    { label: 'Freelancers', href: '/freelancers' },
    { label: 'About', href: '/about' },
    { label: 'Contact', href: '/contact' },
];

function ElevationScroll({ children }: { children: React.ReactElement }) {
    const trigger = useScrollTrigger({
        disableHysteresis: true,
        threshold: 0,
    });

    return React.cloneElement(children, {
        elevation: trigger ? 4 : 0,
        sx: {
            ...children.props.sx,
            backgroundColor: trigger ? 'rgba(255, 255, 255, 0.95)' : 'white',
            backdropFilter: trigger ? 'blur(10px)' : 'none',
            transition: 'all 0.3s ease',
        },
    });
}

export default function Header() {
    const [mobileOpen, setMobileOpen] = useState(false);

    const handleDrawerToggle = () => {
        setMobileOpen(!mobileOpen);
    };

    const drawer = (
        <Box onClick={handleDrawerToggle} sx={{ textAlign: 'center', p: 2 }}>
            <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
                <Box sx={{
                    fontSize: '1.5rem',
                    fontWeight: 700,
                    background: 'linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%)',
                    WebkitBackgroundClip: 'text',
                    WebkitTextFillColor: 'transparent',
                }}>
                    Freeio
                </Box>
                <IconButton onClick={handleDrawerToggle}>
                    <CloseIcon />
                </IconButton>
            </Box>
            <List>
                {navItems.map((item) => (
                    <ListItem key={item.label} disablePadding>
                        <ListItemButton component={Link} href={item.href} sx={{ textAlign: 'center' }}>
                            <ListItemText primary={item.label} />
                        </ListItemButton>
                    </ListItem>
                ))}
                <ListItem disablePadding>
                    <ListItemButton sx={{ textAlign: 'center' }}>
                        <ListItemText primary="Sign In" />
                    </ListItemButton>
                </ListItem>
                <ListItem disablePadding>
                    <Button
                        variant="contained"
                        fullWidth
                        sx={{ m: 1 }}
                    >
                        Sign Up
                    </Button>
                </ListItem>
            </List>
        </Box>
    );

    return (
        <>
            <ElevationScroll>
                <AppBar position="sticky" sx={{ backgroundColor: 'white', color: 'text.primary' }}>
                    <Container maxWidth="xl">
                        <Toolbar disableGutters sx={{ py: 1 }}>
                            {/* Logo */}
                            <Link href="/" style={{ textDecoration: 'none', color: 'inherit' }}>
                                <Box sx={{
                                    fontSize: '1.75rem',
                                    fontWeight: 700,
                                    background: 'linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%)',
                                    WebkitBackgroundClip: 'text',
                                    WebkitTextFillColor: 'transparent',
                                    cursor: 'pointer',
                                    mr: 4,
                                }}>
                                    Freeio
                                </Box>
                            </Link>

                            {/* Desktop Navigation */}
                            <Box sx={{ flexGrow: 1, display: { xs: 'none', md: 'flex' }, gap: 1 }}>
                                {navItems.map((item) => (
                                    <Button
                                        key={item.label}
                                        component={Link}
                                        href={item.href}
                                        sx={{
                                            color: 'text.primary',
                                            '&:hover': {
                                                color: 'primary.main',
                                                backgroundColor: 'rgba(99, 102, 241, 0.08)',
                                            },
                                        }}
                                    >
                                        {item.label}
                                    </Button>
                                ))}
                            </Box>

                            {/* Search Bar - Desktop */}
                            <Box sx={{ display: { xs: 'none', lg: 'block' }, mr: 2 }}>
                                <TextField
                                    size="small"
                                    placeholder="Search services..."
                                    sx={{
                                        width: 250,
                                        '& .MuiOutlinedInput-root': {
                                            backgroundColor: 'rgba(99, 102, 241, 0.05)',
                                        },
                                    }}
                                    InputProps={{
                                        startAdornment: (
                                            <InputAdornment position="start">
                                                <SearchIcon sx={{ color: 'text.secondary' }} />
                                            </InputAdornment>
                                        ),
                                    }}
                                />
                            </Box>

                            {/* Auth Buttons - Desktop */}
                            <Box sx={{ display: { xs: 'none', md: 'flex' }, gap: 1 }}>
                                <Button
                                    variant="text"
                                    sx={{ color: 'text.primary' }}
                                >
                                    Sign In
                                </Button>
                                <Button
                                    variant="contained"
                                >
                                    Sign Up
                                </Button>
                            </Box>

                            {/* Mobile Menu Button */}
                            <Box sx={{ flexGrow: 1, display: { xs: 'flex', md: 'none' }, justifyContent: 'flex-end' }}>
                                <IconButton
                                    color="inherit"
                                    aria-label="open drawer"
                                    edge="start"
                                    onClick={handleDrawerToggle}
                                >
                                    <MenuIcon />
                                </IconButton>
                            </Box>
                        </Toolbar>
                    </Container>
                </AppBar>
            </ElevationScroll>

            {/* Mobile Drawer */}
            <Drawer
                anchor="right"
                open={mobileOpen}
                onClose={handleDrawerToggle}
                ModalProps={{
                    keepMounted: true,
                }}
                sx={{
                    display: { xs: 'block', md: 'none' },
                    '& .MuiDrawer-paper': { boxSizing: 'border-box', width: 280 },
                }}
            >
                {drawer}
            </Drawer>
        </>
    );
}
