'use client';

import React from 'react';
import {
    Card,
    CardContent,
    CardMedia,
    Typography,
    Box,
    Avatar,
    Rating,
    IconButton,
    Chip,
} from '@mui/material';
import { Favorite, FavoriteBorder } from '@mui/icons-material';
import Link from 'next/link';

interface ServiceCardProps {
    id: number;
    title: string;
    category: string;
    image: string;
    provider: {
        name: string;
        avatar: string;
        rating: number;
        reviews: number;
    };
    price: number;
    featured?: boolean;
}

export default function ServiceCard({
    id,
    title,
    category,
    image,
    provider,
    price,
    featured = false,
}: ServiceCardProps) {
    return (
        <Card
            sx={{
                height: '100%',
                display: 'flex',
                flexDirection: 'column',
                position: 'relative',
                cursor: 'pointer',
            }}
        >
            <Link href={`/services/${id}`} style={{ textDecoration: 'none', color: 'inherit' }}>
                <Box sx={{ position: 'relative' }}>
                    <CardMedia
                        component="img"
                        height="200"
                        image={image}
                        alt={title}
                        sx={{
                            transition: 'transform 0.3s ease',
                            '&:hover': {
                                transform: 'scale(1.05)',
                            },
                        }}
                    />
                    {featured && (
                        <Chip
                            label="Featured"
                            size="small"
                            sx={{
                                position: 'absolute',
                                top: 12,
                                left: 12,
                                backgroundColor: 'secondary.main',
                                color: 'white',
                                fontWeight: 600,
                            }}
                        />
                    )}
                    <IconButton
                        sx={{
                            position: 'absolute',
                            top: 8,
                            right: 8,
                            backgroundColor: 'white',
                            '&:hover': {
                                backgroundColor: 'white',
                                color: 'error.main',
                            },
                        }}
                        size="small"
                    >
                        <FavoriteBorder />
                    </IconButton>
                </Box>

                <CardContent sx={{ flexGrow: 1, p: 2.5 }}>
                    <Chip
                        label={category}
                        size="small"
                        sx={{
                            mb: 1.5,
                            backgroundColor: 'rgba(99, 102, 241, 0.1)',
                            color: 'primary.main',
                            fontWeight: 500,
                            fontSize: '0.75rem',
                        }}
                    />

                    <Typography
                        variant="h6"
                        sx={{
                            fontSize: '1rem',
                            fontWeight: 600,
                            mb: 2,
                            display: '-webkit-box',
                            WebkitLineClamp: 2,
                            WebkitBoxOrient: 'vertical',
                            overflow: 'hidden',
                            lineHeight: 1.4,
                        }}
                    >
                        {title}
                    </Typography>

                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1.5, mb: 2 }}>
                        <Avatar
                            src={provider.avatar}
                            alt={provider.name}
                            sx={{ width: 32, height: 32 }}
                        />
                        <Box sx={{ flexGrow: 1 }}>
                            <Typography variant="body2" fontWeight={500}>
                                {provider.name}
                            </Typography>
                            <Box sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
                                <Rating value={provider.rating} precision={0.1} size="small" readOnly />
                                <Typography variant="caption" color="text.secondary">
                                    ({provider.reviews})
                                </Typography>
                            </Box>
                        </Box>
                    </Box>

                    <Box
                        sx={{
                            display: 'flex',
                            justifyContent: 'space-between',
                            alignItems: 'center',
                            pt: 2,
                            borderTop: '1px solid',
                            borderColor: 'divider',
                        }}
                    >
                        <Typography variant="body2" color="text.secondary">
                            Starting at
                        </Typography>
                        <Typography variant="h6" fontWeight={700} color="primary.main">
                            ${price}
                        </Typography>
                    </Box>
                </CardContent>
            </Link>
        </Card>
    );
}
